package ProgolInterface;


import java.awt.*;
import java.io.*;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import myawt.InfoDialog;
import myawt.GridBag;
import PrologParse.*;

/**
 * The Progol Interface Class - An interface for Progol! 
 * This class holds all the subsection panels, and keeps
 * a central record of the session types, modes, clauses etc.
 * @author Rupert Parson
 * @version 2.0
 * @see TypeSelectPanel
 * @see ModeSelectPanel
 * @see ClauseSelectPanel
 * @see ProgolExecPanel
 */
public class ProgolInterface extends Frame implements ActionListener {  
  protected ModeList modes;
  protected ClauseList options;
  protected ClauseList types;
  protected ClauseList clauses;

  private MenuBar menubar;
  private Menu filemenu, helpmenu;
  private MenuItem newSession, loadSession, saveSession, quit;
  private MenuItem help, about;

  private Button next, prev;  
  private Label label;
  private Panel buttonpanel, cardpanel;
  
  private TypeSelectPanel typesPanel;
  private ModeSelectPanel modesPanel;
  private ClauseSelectPanel clausePanel;
  private ProgolExecPanel progolPanel;

  private CardLayout card = new CardLayout(10,10);
  private GridLayout grid = new GridLayout(1,2,20,40);
  private GridBagLayout gridbag = new GridBagLayout();

  private int panel = 1;
  private static String[] labels = {"",
				    "Type Definition",
				    "Mode Definition",
				    "Clause Definition",
				    "Progol Execution",
				    ""};
  /** 
   * Constructor to start the session.
   * The constructor adds in the various section panels, and 
   * adds buttons to flip between them.  All the mode and clause
   * storage objects are initialised to be empty.
   * @param title The title of the session.
   */
  public ProgolInterface(String title) {
    super(title);
    this.setFont(new Font("Helvetica", Font.PLAIN, 12));

    modes = new ModeList();
    types = new ClauseList();
    options = new ClauseList();
    clauses = new ClauseList();

    typesPanel = new TypeSelectPanel(this);
    modesPanel = new ModeSelectPanel(this);
    clausePanel = new ClauseSelectPanel(this);
    progolPanel = new ProgolExecPanel(this);

    cardpanel = new Panel();
    cardpanel.setLayout(card);
    cardpanel.add("North",typesPanel);
    cardpanel.add("North",modesPanel);
    cardpanel.add("North",clausePanel);
    cardpanel.add("North",progolPanel);

    buttonpanel = new Panel();
    buttonpanel.setLayout(grid);
    next = new Button("");
    prev = new Button("");
    next.addActionListener(this);
    prev.addActionListener(this);
    label = new Label();
    label.setAlignment(Label.CENTER);
    buttonpanel.add(prev);
    buttonpanel.add(label);
    buttonpanel.add(next);
    label();

    newSession = new MenuItem("New Session");
    newSession.addActionListener(this);
    loadSession = new MenuItem("Load Session");
    loadSession.addActionListener(this);
    saveSession = new MenuItem("Save Session");
    saveSession.addActionListener(this);
    quit = new MenuItem("Quit");
    quit.addActionListener(this);
    filemenu = new Menu("File");
    filemenu.add(newSession);
    filemenu.add(loadSession);
    filemenu.add(saveSession);
    filemenu.add(quit);

    help = new MenuItem("Help");
    help.addActionListener(this);
    about = new MenuItem("About");
    about.addActionListener(this);
    helpmenu = new Menu("Help");
    helpmenu.add(help);
    helpmenu.add(about);

    menubar = new MenuBar();
    menubar.add(filemenu);
    menubar.add(helpmenu);
    menubar.setHelpMenu(helpmenu);
    this.setMenuBar(menubar);

    this.setLayout(gridbag);
    GridBag.constrain(this, cardpanel, 0, 0, 1, 1, 
	      GridBagConstraints.BOTH, 
	      GridBagConstraints.CENTER,
	      1.0, 1.0, 0, 0, 0, 0);
    GridBag.constrain(this, buttonpanel, 0, 1, 1, 1, 
	      GridBagConstraints.HORIZONTAL, 
	      GridBagConstraints.CENTER,
	      1.0, 0.0, 10, 10, 10, 10);
  }

  /**
   * Update all the panels in this session.
   */
  public final void updateAll() {
    typesPanel.update();
    modesPanel.update();
    clausePanel.update();
  }

  /**
   * Relabel the buttons
   */
  private final void label() {
    prev.setLabel("<<  " + labels[panel-1]);
    label.setText(labels[panel]);
    next.setLabel(labels[panel+1] + "  >>");
    
    if (panel == 1) {
      prev.setEnabled(false);
      next.setEnabled(true);
    }
    else if (panel == labels.length-2) {
      prev.setEnabled(true);
      next.setEnabled(false);
    }
    else {
      prev.setEnabled(true);
      next.setEnabled(true);
    }
  }

  /**
   * Save the session in the given filename.
   * The file is written in the following order:
   * Options, 
   * Modes,
   * Types,
   * Other Clauses (examples and background knowledge.).
   * @param filename The file into which the session is written.
   */
  public final boolean saveSession(String filename) {
    try {
      Enumeration enum;
      FileWriter ws = 
	new FileWriter(filename);
      enum = modes.elements();
      while (enum.hasMoreElements()) {
	ws.write(((Mode) enum.nextElement()).toString() + "\n");
      }
      enum = types.elements();
      while (enum.hasMoreElements()) {
	ws.write(((Clause) enum.nextElement()).toString() + "\n");
      }
      enum = clauses.elements();
      while (enum.hasMoreElements()) {
	ws.write(((Clause) enum.nextElement()).toString() + "\n");
      }
      enum = options.elements();
      while (enum.hasMoreElements()) {
	ws.write(((Clause) enum.nextElement()).toString() + "\n");
      }
      ws.close();
    }
    catch(IOException e) { System.out.println("Whoops: " + e.toString()); }

    return true;
  }

  /**
   * Load a session from the given filename.
   * The file is first scanned for QUERY type clauses - i.e.
   * modes and settings.  Then the file is scanned for types
   * and general clauses, the distinction being made on the basis
   * of the mode declarations (and other more obvious factors).
   * @param filename The file from which the session is loaded.
   */
  public final boolean loadSession(String filename) {
    ClauseList file;
    Clause c;
    Mode m;
    Term t;
    Enumeration e1,e2,e3;
    String s;
    try {
      BufferedReader rs = new BufferedReader(new FileReader(filename));
      file = new PrologParser(rs);

      e1 = file.elements();
      while (e1.hasMoreElements()) {
	c = (Clause) e1.nextElement();

	if (c.punctuation() == Clause.QUERY) {
	  e2 = c.body().elements();
	  while (e2.hasMoreElements()) {
	    t = (Term) e2.nextElement();
	    s = " :- " + t.toString() + "?";
	    if (t.symbol().equals("modeh/2") || t.symbol().equals("modeb/2")) {
	      m = new Mode(new Clause(s));
	      clauses.addDefinition(m.predicateSymbol());
	      e3 = m.arguments().elements();
	      while (e3.hasMoreElements()) {
		String tp = ((ModeArg) e3.nextElement()).type();
		if (!tp.equals("any") &&
		    !tp.equals("int") && 
		    !tp.equals("float")) {
		  types.addDefinition(tp + "/1");
		}
	      }
	      modes.addMode(m);
	    }
	    else {
	      options.addElement(new Clause(s));
	    }
	  }			       
	}
      }

      e1 = file.elements();
      while (e1.hasMoreElements()) {
	c = (Clause) e1.nextElement();

	if (c.punctuation() == Clause.ASSERT) {
	  if ((c.head().functionArity() == 1) && 
	      !modes.hasModeFor(c.head().symbol()) &&
	      c.body().size() ==0) {
	    types.addElement(c);
	  }
	  else {
	    clauses.addElement(c);
	  }
	}
      }
      rs.close();
    }
    catch(IOException e) { System.out.println("Whoops: " + e.toString()); }
    updateAll();
    return true;
  } 

  /**
   * Event handling for buttons and menus.
   */
  public final void actionPerformed(ActionEvent event) {
    if (event.getSource() == next) {
      panel++;
      label();
      card.next(cardpanel);
      updateAll();
    }
    else if (event.getSource() == prev) {
      panel--;
      label();
      card.previous(cardpanel);
      updateAll();
    }
    else if (event.getSource() == newSession) {
      panel = 1;
      label();
      card.first(cardpanel);
      modes.removeAll();
      types.removeAll();
      clauses.removeAll();
      options.removeAll();
      updateAll();
    }
    else if (event.getSource() == loadSession) {
      FileDialog fl = new FileDialog(this, "Load Session", FileDialog.LOAD);
      fl.pack();
      fl.setVisible(true);
      String fn = fl.getFile();
      String fd = fl.getDirectory();
      if (fn != null) {
	File f = new File(fd,fn);
	if (f.canRead()) {
	  loadSession(fd+fn);
	}
      }
      updateAll();
    }
    else if (event.getSource() == saveSession) {
      FileDialog fs = new FileDialog(this, "Save Session", FileDialog.SAVE);
      fs.pack();
      fs.setVisible(true);
      String fn = fs.getFile();
      String fd = fs.getDirectory();
      if (fn != null) {
	File f = new File(fd,fn);
	if (f.canWrite()) {
	  saveSession(fd+fn);
	}
      }
    }
    else if (event.getSource() == quit) {
      System.exit(0);
    }
    else if (event.getSource() == help) {
      Help help = new Help();
      help.setVisible(true);
    }
    else if (event.getSource() == about) {
      InfoDialog info = new InfoDialog(this,"About Progol Inteface", 
				       "Progol Interface Version 2.0",
			    "Copyright (c) 1997 Rupert Parson");
      info.setVisible(true);
    }
  }
  
  /**
   * Create a new ProgolInterface and show it.
   * A single command line parameter may be given, which should
   * be a filename for a session (or prolog) file.
   */  
  public final static void main(String[] args) {
    ProgolInterface session = new ProgolInterface("Progol Interface");
    if (args.length == 1) {
      File f = new File(args[0]);
      if (f.canRead()) {
	session.loadSession(args[0]);
      }
    }
    session.pack();
    session.setVisible(true);
  }
}
